# standard library imports
from datetime import datetime
import logging

# constants
MAX_LOG_SIZE = 1024 ** 2 # 1MB
BACK_UP_COUNT = 2
LOG_FILE = "mission.log"

class PiLogger:
    """
    The class used for logging.
    """

    def __init__(self) -> None:
        """
        Initialize a new PiLogger instance.
        """
        try:
            from logzero import setup_logger

            # Set up formatter
            formatter = logging.Formatter(
                '%(asctime)s [%(levelname)s] - %(message)s'
            )

            # Configure logzero with rotation settings
            self.logger = setup_logger(
                name=f"pi_logger_{datetime.now().strftime('%Y%m%d_%H%M%S')}",
                logfile=LOG_FILE,
                formatter=formatter,
                maxBytes=MAX_LOG_SIZE,
                backupCount=BACK_UP_COUNT,
                disableStderrLogger=False
            )

            self.info(f"PiLogger initialized with log file: {LOG_FILE}")

        except Exception as e:
            # Abort if logging fails.
            print(f"Fatal Error! Could not set up logging using the 'logzero' package into {LOG_FILE}.")
            print("The error was the following: ", e)
            print("Aborting! (exit code 2) //means error")
            raise SystemExit(2)

    def debug(self, message: str) -> None:
        """Log a debug message."""
        self.logger.debug(self.currentOps + message)

    def info(self, message: str) -> None:
        """Log an informational message."""
        self.logger.info(self.currentOps + message)

    def warning(self, message: str) -> None:
        """Log a warning message."""
        self.logger.warning(self.currentOps + message)

    def error(self, message: str) -> None:
        """Log an error message."""
        self.logger.error(self.currentOps + message)

    def critical(self, message: str) -> None:
        """Log a critical message."""
        self.logger.critical(self.currentOps + message)

    def exception(self, message: str) -> None:
        """Log an exception message with traceback."""
        self.logger.exception(self.currentOps + message)

    def log_dict(self, data: dict, level: int = logging.INFO, prefix: str = "") -> None:
        """
        Log a dictionary of values.

        Args:
            data: Dictionary of values to log
            level: Logging level to use
            prefix: Optional prefix for the log message
        """
        try:
            message = prefix + " " if prefix else ""
            message += ", ".join([f"{k}={v}" for k, v in data.items()])
            self.logger.log(level, self.currentOps + message)
        except Exception as e:
            self.warning(self.currentOps + f"Failed to log dictionary: {str(e)}")

    currentOps = ""
    opNameLens = []
    def operation_start(self, operation_name: str) -> None:
        """Mark the start of a significant operation in logs."""
        self.info(f"OPERATION START: {operation_name} at {datetime.now().isoformat()}")
        self.currentOps += operation_name + " > "
        name_len = len(operation_name)
        self.opNameLens.append(name_len)

    def operation_end(self, status: str = "completed") -> None:
        """Mark the end of a significant operation in logs."""
        if self.opNameLens is None or len(self.opNameLens) == 0:
            self.error("Trying to log the end of an operation when no operations have been started.")

        last_len = self.opNameLens[-1] + 3
        self.opNameLens.pop()
        opname = self.currentOps[-last_len:-3]
        self.currentOps = self.currentOps[:-(last_len + 1)]
        self.info(f"OPERATION END: {opname} {status} at {datetime.now().isoformat()}")