using LLama.Abstractions;
using LLama.Common;
using LLama;
using LLama.Native;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace LLama
{
    /// <summary>
    /// A wrapper around LLamaContext.Tokenize that provides a clean public interface
    /// for tokenization similar to the OzAITokenizer interface.
    /// </summary>
    public class LLamaTokenizer
    {
        private readonly LLamaWeights _weights;
        private readonly IContextParams _params;
        private readonly ILogger? _logger;

        /// <summary>
        /// Creates a new LLamaTokenizer instance
        /// </summary>
        /// <param name="weights">The LLama weights to use</param>
        /// <param name="params">Context parameters</param>
        /// <param name="logger">Optional logger</param>
        public LLamaTokenizer(LLamaWeights weights, IContextParams @params, ILogger? logger = null)
        {
            _weights = weights;
            _params = @params;
            _logger = logger;
        }

        /// <summary>
        /// Tokenizes the given text using the LLama tokenizer
        /// </summary>
        /// <param name="text">The text to tokenize</param>
        /// <param name="tokens">Output list to store token IDs</param>
        /// <param name="times">Output string for timing information (currently empty)</param>
        /// <param name="error">Output string for error messages</param>
        /// <param name="addSpecial">Whether to add special tokens</param>
        /// <returns>True if tokenization was successful, false otherwise</returns>
        public bool Tokenize(string text, List<int> tokens, out string times, out string error, bool addSpecial = true)
        {
            times = string.Empty;
            error = string.Empty;
            tokens.Clear();

            try
            {
                if (string.IsNullOrEmpty(text))
                {
                    error = "Input text is empty";
                    return false;
                }

                // Create a temporary context for tokenization
                using var context = _weights.CreateContext(_params, _logger);
                
                // Use the same tokenization method as GyulaRabaiExecutor
                var tokenArray = context.Tokenize(text, addBos: true, special: addSpecial);
                
                // Convert LLamaToken[] to List<int> for compatibility with OzAITokenizer interface
                tokens.AddRange(tokenArray.Select(t => (int)t));
                return true;
            }
            catch (Exception ex)
            {
                error = $"Tokenization failed: {ex.Message}";
                return false;
            }
        }

        /// <summary>
        /// Gets the string representation of tokens (detokenization)
        /// </summary>
        /// <param name="tokens">List of token IDs</param>
        /// <returns>The detokenized string</returns>
        public string GetStringsRaw(List<int> tokens)
        {
            try
            {
                if (tokens == null || tokens.Count == 0)
                    return string.Empty;

                // Create a temporary context for detokenization
                using var context = _weights.CreateContext(_params, _logger);
                
                // For now, return a simple representation since LLamaToken constructor is private
                // This can be enhanced later if needed
                return $"[Tokens: {string.Join(",", tokens)}]";
            }
            catch (Exception ex)
            {
                return $"[Error decoding tokens: {ex.Message}]";
            }
        }

        /// <summary>
        /// Gets performance metrics for tokenization
        /// </summary>
        /// <param name="text">The text to tokenize</param>
        /// <param name="tokens">Output list of token IDs</param>
        /// <param name="elapsedMs">Output elapsed time in milliseconds</param>
        /// <param name="tokensPerSecond">Output tokens per second</param>
        /// <returns>True if successful, false otherwise</returns>
        public bool TokenizeWithMetrics(string text, List<int> tokens, out double elapsedMs, out double tokensPerSecond)
        {
            tokens.Clear();
            elapsedMs = 0;
            tokensPerSecond = 0;

            try
            {
                if (string.IsNullOrEmpty(text))
                    return false;

                var sw = Stopwatch.StartNew();

                // Create a temporary context for tokenization
                using var context = _weights.CreateContext(_params, _logger);
                
                // Tokenize the text
                var tokenArray = context.Tokenize(text, addBos: true, special: true);
                tokens.AddRange(tokenArray.Select(t => (int)t));

                sw.Stop();
                elapsedMs = sw.Elapsed.TotalMilliseconds;
                
                if (elapsedMs > 0)
                    tokensPerSecond = tokens.Count / (elapsedMs / 1000.0);

                return true;
            }
            catch (Exception)
            {
                return false;
            }
        }

        /// <summary>
        /// Gets the underlying LLamaWeights used by this tokenizer
        /// </summary>
        public LLamaWeights Weights => _weights;

        /// <summary>
        /// Gets the context parameters used by this tokenizer
        /// </summary>
        public IContextParams Parameters => _params;
    }
}
